import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/route/route.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/url_container.dart';
import 'package:viserpay/data/controller/savings/savings_history_controller.dart';
import 'package:viserpay/data/repo/saving/saving_repo.dart';
import 'package:viserpay/data/services/api_service.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/custom_loader/custom_loader.dart';
import 'package:viserpay/view/components/no_data.dart';
import 'package:viserpay/view/screens/savings/widget/dps_history_widget.dart';

class SavingsHistoryScreen extends StatefulWidget {
  const SavingsHistoryScreen({super.key});

  @override
  State<SavingsHistoryScreen> createState() => _SavingsHistoryScreenState();
}

class _SavingsHistoryScreenState extends State<SavingsHistoryScreen> with SingleTickerProviderStateMixin {
  late TabController _tabController;

  List<String> title = [
    MyStrings.running,
    MyStrings.matured,
    MyStrings.closed,
  ];
  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(SavingRepo(apiClient: Get.find()));
    final controller = Get.put(SavingsHistoryController(repo: Get.find()));
    _tabController = TabController(length: 3, vsync: this);

    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((t) {
      controller.initialize();
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: MyColor.screenBgColor,
      appBar: CustomAppBar(title: MyStrings.savingsHistory, isTitleCenter: true),
      body: GetBuilder<SavingsHistoryController>(
        builder: (controller) {
          return Padding(
            padding: Dimensions.screenPaddingHV,
            child: Column(
              children: [
                Container(
                  decoration: BoxDecoration(color: MyColor.colorWhite, borderRadius: BorderRadius.circular(Dimensions.defaultRadius)),
                  child: TabBar(
                    controller: _tabController,
                    labelColor: MyColor.primaryColor,
                    dividerColor: MyColor.getContentTextColor().withValues(alpha: 0.1),
                    unselectedLabelColor: MyColor.getContentTextColor(),
                    indicatorColor: MyColor.primaryColor,
                    isScrollable: true,
                    tabAlignment: TabAlignment.center,
                    labelPadding: EdgeInsetsDirectional.symmetric(vertical: Dimensions.space5, horizontal: Dimensions.space10),
                    tabs: List.generate(title.length, (index) => Tab(text: title[index].tr)),
                    onTap: (value) {},
                  ),
                ),
                controller.isLoading
                    ? Expanded(child: CustomLoader(isFullScreen: true))
                    : Expanded(
                        flex: 1,
                        child: TabBarView(
                          physics: AlwaysScrollableScrollPhysics(parent: BouncingScrollPhysics()),
                          controller: _tabController,
                          viewportFraction: 1,
                          children: [
                            RunningList(),
                            MaturedList(),
                            ClosedList(),
                          ],
                        ),
                      ),
              ],
            ),
          );
        },
      ),
    );
  }
}

class RunningList extends StatelessWidget {
  const RunningList({super.key});

  @override
  Widget build(BuildContext context) {
    return GetBuilder<SavingsHistoryController>(
      builder: (controller) {
        return controller.runningDpsList.isEmpty
            ? NoDataWidget(noDataText: MyStrings.noSavingsAccountFound)
            : ListView.separated(
                separatorBuilder: (context, index) => SizedBox(height: Dimensions.space10),
                itemCount: controller.runningDpsList.length,
                itemBuilder: (context, index) {
                  return DpsHistoryWidget(
                    dps: controller.runningDpsList[index],
                    currency: controller.repo.apiClient.getCurrencyOrUsername(isSymbol: true),
                    imagePath: "${UrlContainer.domainUrl}/${controller.organizationImagePath}/${controller.runningDpsList[index].dpsPlan?.organization?.image}",
                    onTap: () {
                      Get.toNamed(
                        RouteHelper.installmentScreen,
                        arguments: controller.runningDpsList[index].id,
                      );
                    },
                  );
                },
              );
      },
    );
  }
}

class MaturedList extends StatelessWidget {
  const MaturedList({super.key});

  @override
  Widget build(BuildContext context) {
    return GetBuilder<SavingsHistoryController>(
      builder: (controller) {
        return controller.matureDpsList.isEmpty
            ? NoDataWidget(noDataText: MyStrings.noMatureAccountFound)
            : ListView.separated(
                separatorBuilder: (context, index) => SizedBox(height: Dimensions.space10),
                itemCount: controller.matureDpsList.length,
                itemBuilder: (context, index) {
                  return DpsHistoryWidget(
                    dps: controller.matureDpsList[index],
                    currency: controller.repo.apiClient.getCurrencyOrUsername(isSymbol: true),
                    imagePath: "${UrlContainer.domainUrl}/${controller.organizationImagePath}/${controller.matureDpsList[index].dpsPlan?.organization?.image}",
                    onTap: () {
                      Get.toNamed(RouteHelper.installmentScreen, arguments: controller.matureDpsList[index].id);
                    },
                  );
                },
              );
      },
    );
  }
}

class ClosedList extends StatelessWidget {
  const ClosedList({super.key});

  @override
  Widget build(BuildContext context) {
    return GetBuilder<SavingsHistoryController>(
      builder: (controller) {
        return controller.closedDpsList.isEmpty
            ? NoDataWidget(noDataText: MyStrings.noClosedAccountFound)
            : ListView.separated(
                separatorBuilder: (context, index) => SizedBox(height: Dimensions.space10),
                itemCount: controller.closedDpsList.length,
                itemBuilder: (context, index) {
                  return DpsHistoryWidget(
                    dps: controller.closedDpsList[index],
                    currency: controller.repo.apiClient.getCurrencyOrUsername(isSymbol: true),
                    imagePath: "${UrlContainer.domainUrl}/${controller.organizationImagePath}/${controller.closedDpsList[index].dpsPlan?.organization?.image}",
                    onTap: () {
                      Get.toNamed(RouteHelper.installmentScreen, arguments: controller.closedDpsList[index].id);
                    },
                  );
                },
              );
      },
    );
  }
}
